"""Path descriptors to different web hosting-related files and directories

These descriptors are abstract, which means they could have no absolute path: you should convert them
to concrete absolute paths on concrete server. That is necessary because different servers have different location
of data, for example one server could have virtual hosts at '/var/www/vhosts', another could have it at '/opt/vhosts'
"""


class WebHostingPath(object):
	"""Base class for all web hosting-related paths"""
	pass


class AbsolutePath(WebHostingPath):
	"""Absolute path to web hosting-related file or directory"""
	def __init__(self, absolute_path):
		"""
		:type absolute_path: basestring
		"""
		self._absolute_path = absolute_path

	@property
	def absolute_path(self):
		"""
		:rtype: str | unicode
		"""
		return self._absolute_path

	def __eq__(self, other):
		return (
			isinstance(other, AbsolutePath) and
			self.absolute_path == other.absolute_path
		)

	def __repr__(self):
		return '%s(%r)' % (self.__class__.__name__, self.absolute_path)


class VirtualHostsPath(WebHostingPath):
	"""Path to web hosting related file or directory, relative to virtual hosts root directory on the server

	Note that virtual hosts root directory is different for different servers and different control panels.
	"""

	def __init__(self, relative_path):
		"""
		:type relative_path: basestring
		"""
		self._relative_path = relative_path

	@property
	def relative_path(self):
		"""
		:rtype: str | unicode
		"""
		return self._relative_path

	def __eq__(self, other):
		return (
			isinstance(other, VirtualHostsPath) and
			self.relative_path == other.relative_path
		)

	def __repr__(self):
		return '%s(%r)' % (self.__class__.__name__, self.relative_path)


class WebspacePath(WebHostingPath):
	"""Path to web hosting-related file or directory related to webspace"""
	def __init__(self, webspace):
		"""
		:type webspace: parallels.core.dump.data_model.Subscription
		"""
		self._webspace = webspace

	@property
	def webspace(self):
		"""
		:rtype: parallels.core.dump.data_model.Subscription
		"""
		return self._webspace

	def __eq__(self, other):
		return (
			isinstance(other, WebspacePath) and
			other.webspace.name == self.webspace.name
		)

	def __repr__(self):
		return '%s(webspace=%r)' % (self.__class__.__name__, self.webspace.name)


class WebspaceRoot(WebspacePath):
	"""Path to webspace root - top directory available for customer. For Plesk it is /var/www/vhosts/<webspace-name>"""
	pass


class WebspaceDocumentRoot(WebspacePath):
	"""Path to document root of webspace - directory available by HTTP"""
	pass


class WebspaceSSLDocumentRoot(WebspacePath):
	"""Path to secure document root of webspace - directory available by HTTPS"""
	pass


class WebspaceCGIBin(WebspacePath):
	"""Path to cgi-bin directory of webspace"""
	pass


class WebspaceStatistics(WebspacePath):
	"""Path to directory with webspace statistics files (AWStats/Webalizer/etc)"""
	pass


class WebspaceMainDomainStatistics(WebspacePath):
	"""Path to directory with webspace main site's statistics files (AWStats/Webalizer/etc)"""
	pass


class WebspaceSecurityFile(WebspacePath):
	"""Path to Plesk's .Security file"""
	pass


class WebspaceAdditionalSecurityFile(WebspacePath):
	def __init__(self, webspace, security_file_path):
		super(WebspaceAdditionalSecurityFile, self).__init__(webspace)
		self._security_file_path = security_file_path

	@property
	def security_file_path(self):
		return self._security_file_path


class WebspaceLogs(WebspacePath):
	"""Path to directory with webspace logs (Apache access and error logs, etc)"""
	pass


class WebspaceProtectedDirs(WebspacePath):
	"""Path to directory with protected directory settings"""
	pass


class WebspaceAnonFTPPub(WebspacePath):
	"""Path to public anonymous FTP directory of webspace"""
	pass


class WebspaceAnonFTPIncoming(WebspacePath):
	"""Path to incoming directory of anonymous FTP of webspace"""
	pass


class WebspaceWebUser(WebspacePath):
	"""Path to web user's directory of webspace (available by HTTP by at '/~web_user_name')"""
	def __init__(self, webspace, webuser_name):
		"""
		:type webspace: parallels.core.dump.data_model.Subscription
		:type webuser_name: basestring
		"""
		super(WebspaceWebUser, self).__init__(webspace)
		self._webuser_name = webuser_name

	@property
	def webuser_name(self):
		"""
		:rtype: basestring
		"""
		return self._webuser_name

	def __eq__(self, other):
		return (
			isinstance(other, WebspaceWebUser) and
			other.webspace.name == self.webspace.name and
			other.webuser_name == self.webuser_name
		)

	def __repr__(self):
		return '%s(webspace=%r, webuser_name=%r)' % (self.__class__.__name__, self.webspace.name, self.webuser_name)


class WebspaceMainDomainPrivate(WebspacePath):
	"""Path to directory with private files of main domain in subscription"""
	pass


class WebspacePathTemplate(WebspacePath):
	"""Path template for webspace, with variables that will be expanded"""
	def __init__(self, webspace, template):
		"""
		:type webspace: parallels.core.dump.data_model.Subscription
		:type template: basestring
		"""
		super(WebspacePathTemplate, self).__init__(webspace)
		self._template = template

	@property
	def template(self):
		return self._template

	def __eq__(self, other):
		return (
			isinstance(other, WebspacePathTemplate) and
			other.webspace.name == self.webspace.name and
			other.template == self.template
		)

	def __repr__(self):
		return '%s(webspace=%r, template=%r)' % (self.__class__.__name__, self.webspace.name, self.template)


class SitePath(WebHostingPath):
	"""Path to web hosting-related file or directory related to site (addon domain or subdomain)"""
	def __init__(self, webspace, site):
		"""
		:type webspace: parallels.core.dump.data_model.Subscription
		:type site: parallels.core.dump.data_model.AddonDomain
		"""
		self._webspace = webspace
		self._site = site

	@property
	def site(self):
		"""
		:rtype: parallels.core.dump.data_model.AddonDomain
		"""
		return self._site

	@property
	def webspace(self):
		"""
		:rtype: parallels.core.dump.data_model.Subscription
		"""
		return self._webspace

	def __eq__(self, other):
		return (
			isinstance(other, SitePath) and
			other.webspace.name == self.webspace.name and
			other.site.name == self.site.name
		)

	def __repr__(self):
		return '%s(webspace=%r, site=%r)' % (self.__class__.__name__, self.webspace.name, self.site.name)


class SiteDocumentRoot(SitePath):
	"""Path to document root of webspace - directory available by HTTP"""
	pass


class SiteCGIBin(SitePath):
	"""Path to cgi-bin directory of webspace"""
	pass


class SiteStatistics(SitePath):
	"""Path to directory with webspace statistics files (AWStats/Webalizer/etc)"""
	pass


class SiteLogs(SitePath):
	"""Path to directory with webspace logs (Apache access and error logs, etc)"""
	pass


class SiteProtectedDirs(SitePath):
	"""Path to directory with protected directory settings"""
	pass


class SitePrivate(SitePath):
	"""Path to directory with private files of a site"""
	pass


class SitePathTemplate(SitePath):
	"""Path template for site, with variables that will be expanded"""
	def __init__(self, webspace, site, template):
		"""
		:type webspace: parallels.core.dump.data_model.Subscription
		:type site: parallels.core.dump.data_model.AddonDomain
		:type template: basestring
		"""
		super(SitePathTemplate, self).__init__(webspace, site)
		self._template = template

	@property
	def template(self):
		return self._template

	def __eq__(self, other):
		return (
			isinstance(other, SitePathTemplate) and
			other.webspace.name == self.webspace.name and
			other.site.name == self.site.name and
			other.template == self.template
		)

	def __repr__(self):
		return '%s(webspace=%r, site=%r, template=%r)' % (
			self.__class__.__name__, self.webspace.name, self.site.name, self.template
		)