from parallels.core.actions.content.web.copy_windows_content_base import CopyWindowsWebContentBase
from parallels.core.utils.paths import web_paths
from parallels.core.utils.paths.copy_web_content import CopyWebContentItem
from parallels.plesk.source.plesk.actions.content.web.utils import list_security_files


class CopyWindowsWebContent(CopyWindowsWebContentBase):
	def _list_files_to_copy(self, global_context, subscription):
		"""Make a list of source server directories to be transferred.

		Return a list of (source directory -> destination directory) mappings.

		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:type subscription: parallels.core.migrated_subscription.MigratedSubscription
		:rtype: list[parallels.core.utils.paths.copy_web_content.CopyWebContentItem]
		"""
		vhost_name = subscription.name_idn
		source_server = subscription.web_source_server
		""":type: parallels.plesk.source.plesk.server.PleskSourceServer"""

		security_files = list_security_files(subscription, include_subdomains=False)

		tocopy = [
			# content
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(vhost_name + '/'),
				target_path=web_paths.WebspaceRoot(subscription.converted_dump),
				exclude=[u"/statistics", u"/subdomains"] + ["/%s" % security_file for security_file in security_files],
				skip_if_source_not_exists=False
			),
			# subscription logs
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/statistics/logs/" % vhost_name),
				target_path=web_paths.WebspaceLogs(subscription.converted_dump),
				exclude=[],
				skip_if_source_not_exists=True
			),
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/logs/" % vhost_name),
				target_path=web_paths.WebspaceLogs(subscription.converted_dump),
				exclude=[],
				skip_if_source_not_exists=True
			),
			# subscription statistics
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/statistics/" % vhost_name),
				target_path=web_paths.WebspaceMainDomainStatistics(subscription.converted_dump),
				exclude=[u"/logs"],
				skip_if_source_not_exists=True
			),
			# ".Security" files
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/.Security" % vhost_name),
				target_path=web_paths.WebspaceSecurityFile(subscription.converted_dump),
				exclude=[],
				skip_if_source_not_exists=True
			),
			CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/.plesk/.Security" % vhost_name),
				target_path=web_paths.WebspaceSecurityFile(subscription.converted_dump),
				exclude=[],
				skip_if_source_not_exists=True
			),
		]

		for security_file in security_files:
			tocopy.append(CopyWebContentItem(
				source_path=web_paths.VirtualHostsPath(u"%s/%s" % (vhost_name, security_file)),
				target_path=web_paths.WebspaceAdditionalSecurityFile(subscription.converted_dump, security_file),
				exclude=[],
				skip_if_source_not_exists=True
			))

		for site in subscription.converted_dump.iter_sites():
			site_vhost_name = site.name.encode('idna')

			if source_server.plesk_version >= (12, 0):  # Plesk 12.x
				tocopy.extend([
					# site statistics
					CopyWebContentItem(
						source_path=web_paths.VirtualHostsPath(
							u"%s/.plesk/statistics/%s" % (vhost_name, site_vhost_name)
						),
						target_path=web_paths.SiteStatistics(subscription.converted_dump, site),
						exclude=[],
						skip_if_source_not_exists=True
					),
				])
			elif source_server.plesk_version >= (10, 0):  # Plesk 10 and 11
				source_logs_subdir = u"%s/statistics/logs" % site_vhost_name
				tocopy.extend([
					# site logs
					CopyWebContentItem(
						source_path=web_paths.VirtualHostsPath(source_logs_subdir),
						target_path=web_paths.WebspaceLogs(subscription.converted_dump),
						exclude=[],
						skip_if_source_not_exists=True
					),
					# site statistics
					CopyWebContentItem(
						source_path=web_paths.VirtualHostsPath(u"%s/statistics" % site_vhost_name),
						target_path=web_paths.SiteStatistics(subscription.converted_dump, site),
						exclude=[u'/logs'],
						skip_if_source_not_exists=True
					),
				])
			else:
				# Plesk 8 and 9
				if site.hosting_type == 'vrt_hst':
					# subdomains with physical hosting (living under subdomains/)
					# have separate stats and logs, that need to be copied
					source_site_subdir = "%s/subdomains/%s/" % (vhost_name, site.short_name.encode('idna'))
					source_stats_subdir = u"%sstatistics/" % source_site_subdir
					source_logs_subdir = u"%sstatistics/logs/" % source_site_subdir
					tocopy.extend([
						# subdomain logs
						CopyWebContentItem(
							source_path=web_paths.VirtualHostsPath(source_logs_subdir),
							target_path=web_paths.WebspaceLogs(subscription.converted_dump),
							exclude=[],
							skip_if_source_not_exists=True
						),
						# subdomain stats
						CopyWebContentItem(
							source_path=web_paths.VirtualHostsPath(source_stats_subdir),
							target_path=web_paths.SiteStatistics(subscription.converted_dump, site),
							exclude=[u'/logs'],
							skip_if_source_not_exists=True
						),
					])
					# copy all remaining stuff as content
					tocopy.extend([
						CopyWebContentItem(
							source_path=web_paths.VirtualHostsPath(source_site_subdir),
							target_path=web_paths.WebspacePathTemplate(
								subscription.converted_dump,
								'{webspace_root}/subdomains/%s' % site.short_name.encode('idna')
							),
							exclude=[u'/statistics'],
							skip_if_source_not_exists=False
						)
					])
				else:
					# Subdomains with subdir hosting (living under httpdocs/)
					# do not have separate stats, but have separate log
					# Logs will be copied automatically with subscription's logs
					pass

		return tocopy

	def _get_source_vhosts_dir(self, global_context, source_server):
		"""Get directory where virtual hosts data is located.

		:type global_context: parallels.core.global_context.GlobalMigrationContext
		:rtype: str
		"""
		return source_server.vhosts_dir